<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use App\Models\ExpensesModel;
use App\Models\ExpenseCategory;

class MobExpenseController extends Controller
{
  public function index(Request $request)
{
    $query = ExpensesModel::with('category')->where('branch_id',$request->branch_id);

    if ($request->has('search') && !empty($request->search)) {
        $searchTerm = $request->search;
        $query->where(function ($q) use ($searchTerm) {
            $q->where('expense_name', 'like', "%$searchTerm%")
              ->orWhereHas('category', function ($q2) use ($searchTerm) {
                  $q2->where('category_name', 'like', "%$searchTerm%");
              });
        });
    }

    $pageSize = $request->get('pageSize', 10); 
    $expenses = $query->orderBy('id', 'desc')->paginate($pageSize);

    return response()->json($expenses, 200);
}



    public function categories()
    {
        $categories = ExpenseCategory::all();
        return response()->json($categories, 200);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'expense_category_id' => 'required|exists:expense_category,id',
            'expense_name' => 'required|string',
            'expense_amount' => 'required|numeric',
            'expense_date' => 'required|date',
            'attachment' => 'nullable|file',
        ]);

        DB::beginTransaction();
        try {
            $attachmentPath = null;

            if ($request->hasFile('attachment')) {
                $attachmentPath = storeFile($request->file('attachment'), 'expense');
            }

            $expense = ExpensesModel::create([
                'expense_category_id' => $validated['expense_category_id'],
                'expense_name' => $validated['expense_name'],
                'expense_amount' => $validated['expense_amount'],
                'expense_date' => $validated['expense_date'],
                'expense_attachment' => $attachmentPath,
                'branch_id' => $request->branch_id, 
            ]);

            DB::commit();

            return response()->json(['message' => 'Expense created successfully', 'data' => $expense], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Mobile Expense Store Error: ' . $e->getMessage());

            return response()->json(['message' => 'Failed to create expense'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'expense_category_id' => 'required|exists:expense_category,id',
            'expense_name' => 'required|string',
            'expense_amount' => 'required|numeric',
            'expense_date' => 'required|date',
            'attachment' => 'nullable|file',
        ]);

        DB::beginTransaction();
        try {
            $expense = ExpensesModel::findOrFail($id);

            if ($request->hasFile('attachment')) {
                $attachmentPath = storeFile($request->file('attachment'), 'expense');

                if ($expense->expense_attachment && Storage::disk('s3')->exists($expense->expense_attachment)) {
                    Storage::disk('s3')->delete($expense->expense_attachment);
                }
            } else {
                $attachmentPath = $expense->expense_attachment;
            }

            $expense->update([
                'expense_category_id' => $validated['expense_category_id'],
                'expense_name' => $validated['expense_name'],
                'expense_amount' => $validated['expense_amount'],
                'expense_date' => $validated['expense_date'],
                'expense_attachment' => $attachmentPath,
            ]);

            DB::commit();

            return response()->json(['message' => 'Expense updated successfully', 'data' => $expense], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Mobile Expense Update Error: ' . $e->getMessage());

            return response()->json(['message' => 'Failed to update expense'], 500);
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $expense = ExpensesModel::findOrFail($id);

            if ($expense->expense_attachment && Storage::disk('s3')->exists($expense->expense_attachment)) {
                Storage::disk('s3')->delete($expense->expense_attachment);
            }

            $expense->delete();

            DB::commit();

            return response()->json(['message' => 'Expense deleted successfully'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Mobile Expense Delete Error: ' . $e->getMessage());

            return response()->json(['message' => 'Failed to delete expense'], 500);
        }
    }

       public function expenseCategory()
    {
        $categories = ExpenseCategory::all();
        return response()->json($categories);
    }
}
